(in-package "CL-USER")

; this file contains functions that unroll a description into the
; final formula that we will convert to cnf. imagine init, trans, and
; spec as functions:

; init(S) takes in a state, S, (values for all the variables in the
; desc), and returns true iff it corresponds to an initial state in
; our machine.

; trans(S0,S1) takes two states and returns true if, when given S0, S1
; can result from running the machine for one step.
;spec is a temporal logic formula that takes into account an entire
; run of the machine. however, if we take out the temporal operator,
; what is left applies to two states, this state and the next state,
; and tells whether, for that step, the machine satisfies the
; spec. call the version of spec that has been stripped of the
; temporal operators spec'.

; now, we want to create a formula that is true if the spec is false,
; so that sat will find a counter-example if we have a bug. thus, our
; unrolled formula is:

; init(S0) /\ trans(S0,S1) /\ trans(S1,S2) /\ ... /\ trans(S{k-1},Sk) /\
; (~spec'(S0,S1) \/ ~spec'(S1,S2) \/ ... \/ ~spec'(S{k-1},Sk)

; where k is the number of steps we want to run and ~ corresponds to
; not. so, as long as the transition relation is total (you don't get
; stuck before reaching k steps), then a counter-example will
; correspond to the negation of the spec at one step. oh, and the
; above is only correct if our temporal operators were AG. we were
; focussing on AG because of our plan to distribute the model
; checking. so, you will probably want to add in the other
; operators. also note that spec has already been negated during
; simplification, so actually the temporal operator should be
; EF. there is also a function for handling EG (originally AF), but
; i'm not sure if it is right anymore.

;; (load "dbmc-structs")
;; (load "helper-functions")

;; (load "simplify")

;; (proclaim '(special *cc*))
(defvar *cc* 0)
(defvar *uvars* nil)

(declaim (ftype (function (form-vec fixnum) form-vec) step-form-aux))
(defun step-form-aux (form step)
  (cond ((vec-p form)
	 (let* ((nb (vec-num-bits form))
		(vec (new-vec nb)))
	   (dotimes (i nb (make-unique-vec vec))
	     (vec-set-bit vec i (step-form-aux (vec-get-bit form i) step)))))
	((formula-slot1 form) (formula-slot1 form))
        (t
	 (setf (formula-slot1 form)
	       (case (formula-fn form)
		 (const form)
		 (var (let* ((args (formula-args form))
			     (nvar (make-unique-formula :fn 'var
							:type (formula-type form)
							:args (list (first args)
								    (case (second args)
								      (this step)
								      (next (1+ step))
								      (otherwise (second args)))
								    (if (eq (first (formula-type form)) 'mem)
									'mem
								      (third args))))))
			(setf *uvars* (cons nvar *uvars*))
			nvar))
		 (next      (step-form-aux (car (formula-args form)) (1+ step)))
		 (and       (sb-and-form (mapcar #'(lambda (x) (step-form-aux x step))
						 (formula-args form))))
		 (<->       (sb-equiv-form (mapcar #'(lambda (x) (step-form-aux x step))
						   (formula-args form))))
		 (otherwise (simplify1-step (formula-fn form)
					    (formula-type form)
					    (mapcar (lambda (x)
						      (if (typep x 'form-vec)
							  (step-form-aux x step)
							x))
						    (formula-args form))
					    nil nil nil)))))))

;; (make-unique-formula :fn (formula-fn form)
;; 						 :type (formula-type form)
;; 						 :args (mapcar #'(lambda (x) (step-form-aux x step))
;; 							       (formula-args form)))))))))

;the function you call to apply form to the given step.
(declaim (ftype (function (formula fixnum) formula) step-form))
(defun step-form (form step)
  (if (eq (formula-fn form) 'const)
      form
    (let ((nform (the formula (step-form-aux form step))))
      (clear-slot1 form)
      nform)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; unrolling                  ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;unrolls a formula for the given number of steps. it does this by
;repeatedly applying step-form.
(declaim (ftype (function (formula fixnum) formula) unroll))
(defun unroll (form steps)
  (let ((nforms nil))
    (dotimes (i steps (sb-and-form nforms))
      (setf nforms (cons (step-form form i) nforms)))))

;we or the steps together, since the spec has been negated, and we are simply searching
;for any one step that gives us a counterexample.
(declaim (ftype (function (formula fixnum) formula) unroll-spec-EF))
(defun unroll-spec-EF (spec steps)
  (let ((nforms nil))
    (dotimes (i steps (clear-slot2 (sb-or-form nforms)))
      (setf nforms (cons (step-form spec i) nforms)))))

;here we and the steps together, since the original spec must not hold for every step
;in order to give a counter-example.
(declaim (ftype (function (formula fixnum) formula) unroll-spec-EG))
(defun unroll-spec-EG (spec steps)
  (let ((nforms nil))
    (dotimes (i steps (clear-slot2 (sb-and-form nforms)))
      (setf nforms (cons (step-form spec i) nforms)))))

;unrolling a spec based on its temporal operator.
(declaim (ftype (function (formula fixnum) formula) unroll-spec))
(defun unroll-spec (spec steps)
  (let ((spec-step (the formula (car (formula-args spec)))))
    (if (eq (formula-fn spec) 'EG)
	(unroll-spec-EG spec-step steps)
      (unroll-spec-EF spec-step steps))))

;helper function for contains-next
(declaim (ftype (function (form-vec) boolean) contains-next-aux))
(defun contains-next-aux (form)
  (cond ((vec-p form)
	 (dotimes (i (vec-num-bits form) nil)
	   (when (contains-next-aux (vec-get-bit form i))
	     (return t))))
        ((formula-slot1 form) nil) ;we've already visited this one
        ((eq (formula-fn form) 'var)
	 (eq (second (formula-args form)) 'next))
	((atomic-formp form) nil)
	(t (setf (formula-slot1 form) t)
           (dolist (a (formula-args form) nil)
	     (when (and (typep a 'form-vec)
			(contains-next-aux a) )
               (return t))))))

;tells if form contains a next or not.
(declaim (ftype (function (form-vec) boolean) contains-next))
(defun contains-next (form)
  (let ((cn (contains-next-aux form)))
    (clear-slot1 form)
    cn))

;unrolls desc for the given number of steps, returning the unrolled formula.
(declaim (ftype (function (desc fixnum) (values list formula)) unroll-desc))
(defun unroll-desc (desc steps)
  (setf *uvars* nil)
  ;; (unless (member (formula-fn (desc-spec desc)) '(EG EF)) (setf steps 1))
  (let* ((init (desc-init desc))
	 (trans (desc-trans desc))
	 (spec (desc-spec desc))
	 ;; (vcache (make-hash-table :test 'equal :rehash-size 2.0))
	 (cn (contains-next spec))
	 (trans-steps (cond ((member (formula-fn (desc-spec desc)) '(EG EF))
			     steps)
			    (cn 1)
			    (t 0)))
	 (spec-steps (if (contains-next spec) trans-steps (1+ trans-steps))))
    (let ((uform (sb-and-form (list (unroll init 1)
				    (unroll trans trans-steps)
				    (if (member (formula-fn spec) '(EG EF))
					(unroll-spec spec spec-steps)
				      (unroll spec spec-steps))))))
      (values *uvars*
	      (clear-both-slots uform)))))
